--agora vem o parser
--assim como o lexer, ele também caminha, mas não em caracteres, mas sim em tokens
--passa de token em token usando um cursor como o lexer faz

--baseado no tipo de token e comportamento é criado um node(nó)

--um exemplo de node de variavel
--tokens feitos pelo lexer
--{Var}
--{Identifier, nome}
--{String, kelvi}
--nodes feitos pelo parser
--{Var, nome, LiteralString{kelvi}}

--e assim ele vai organizando a árvore(AST)

--aqui também é emitido alguns erros de "escrita"

--aqui existe muitas funções
--funções usadas

--or
--and
--comparision
--addition
--multiply
--access
--identifier
--primary
--statement

--cada função dessa funciona como precedencia
--primeiro vem a "statement" que processa declarações como variavel/funções/classes/...

--um exemplo da declaração de variavel
--codigo exemplo: var nome "kelvi"
--será executada o "statement", é identificado que é uma declaração de variavel
--guarda o nome da variavel que é um token {Identifier, nome}
--depois é chamado a função "or" primeiro vem essa porque é a que tem a menor precedencia
--o fluxo de chamadas são: or -> and -> comparision -> addition -> multiply -> primary
--o "primary" é quem identifica se é um booleano/número/texto(string)/referencia a variavel/chamada de funções/expressão agrupadas/...
--nesse caso o "primary" encontrou um texto(string) e ele criou o LiteralString{kelvi}
--no final será produzido o node Var{nome, LiteralString{kelvi}}

--o parser não executa o código
--ele apenas organiza os tokens em uma árvore de nodes

local Node = require("node")

---@class Parser
---@field __type "Parser"
---@field tokens Token[] --os tokens que serão processados
---@field cursor integer --a posição do token em "tokens"
local Parser = {}
Parser.__index = Parser
Parser.__type = "Parser"
---cria um novo parser
---@return Parser
function Parser.new()
    return setmetatable({
        tokens = {},
        cursor = 1
    }, Parser)
end
---isso checa se é possivel mover/consumir o token atual
---@return boolean
function Parser:canConsume()
    return self.cursor <= #self.tokens
end
---isso consome o token atual, move para o próximo, atribuindo 1 a "cursor"
function Parser:consume()
    self.cursor = self.cursor + 1
end
---isso obtem o token atual usando o "cursor"
---@return Token
function Parser:peek()
    return self.tokens[self.cursor]
end
---isso checa se o token atual é esperado/possui algum dos o "kinds"
---@param ... TokenKind
function Parser:check(...)
    local token = self:peek()
    for _, kind in ipairs({...}) do
        if token.kind == kind then
            return true
        end
    end
    return false
end
---aqui processa o or
---@return boolean, string|Node
function Parser:parseOr()
    local ok, left = self:parseAnd()
    if not ok then
        return false, left
    end

    while self:canConsume() and self:check("Or") do
        self:consume()

        local ok, right = self:parseAnd()
        if not ok then
            return false, right
        end

        left = Node.new("LogicalExpression", {
            left = left,
            operator = "||",
            right = right
        })
    end

    return true, left
end
---aqui processa o and
---@return boolean, string|Node
function Parser:parseAnd()
    local ok, left = self:parseComparision()
    if not ok then
        return false, left
    end

    while self:canConsume() and self:check("And") do
        self:consume()

        local ok, right = self:parseComparision()
        if not ok then
            return false, right
        end

        left = Node.new("LogicalExpression", {
            left = left,
            operator = "&&",
            right = right
        })
    end

    return true, left
end
---aqui processa as comparações, precedencia de ==, >, >=, <, <=
---@return boolean, string|Node
function Parser:parseComparision()
    local ok, left = self:parseAddition()
    if not ok then
        return false, left
    end

    while self:canConsume() and (self:check("Equal", "Greater", "GreaterEqual", "Less", "LessEqual")) do
        local token = self:peek()
        self:consume()

        local operator = token.kind == "Equal" and '==' or
            token.kind == "Greater" and '>' or
            token.kind == "GreaterEqual" and '>=' or
            token.kind == "Less" and '<' or
            token.kind == "LessEqual" and '<='

        local ok, right = self:parseAddition()
        if not ok then
            return false, right
        end

        left = Node.new("ComparisionExpression", {
            left = left,
            operator = operator,
            right = right
        })
    end

    return true, left
end
---aqui processa a precedencia de + e -
---@return boolean, string|Node
function Parser:parseAddition()
    local ok, left = self:parseMultiply()
    if not ok then
        return false, left
    end

    while self:canConsume() and (self:check("Sum", "Minus")) do
        local token = self:peek()
        self:consume()

        local operator = token.kind == "Sum" and '+' or token.kind == "Minus" and '-'

        local ok, right = self:parseMultiply()
        if not ok then
            return false, right
        end

        left = Node.new("ArithmeticExpression", {
            left = left,
            operator = operator,
            right = right
        })
    end

    return true, left
end
---aqui processa a precedencia de * e /
---@return boolean, string|Node
function Parser:parseMultiply()
    local ok, left = self:parsePrimary()
    if not ok then
        return false, left
    end

    while self:canConsume() and (self:check("Asterisk", "Slash")) do
        local token = self:peek()
        self:consume()

        local operator = token.kind == "Asterisk" and '*' or token.kind == "Slash" and '/'

        local ok, right = self:parsePrimary()
        if not ok then
            return false, right
        end

        left = Node.new("ArithmeticExpression", {
            left = left,
            operator = operator,
            right = right
        })
    end

    return true, left
end
---processando acessos de membro
---@param base Node
---@return boolean, string|Node
function Parser:parseAccess(base)
    while self:canConsume() do
        local symbol = self:peek()
        if symbol.kind == "OpenBracket" then
            self:consume()

            if not self:canConsume() then
                return false, "esperado uma expressão para ser usada como índice"
            end

            local ok, expression = self:parseOr()
            if not ok then
                return false, expression
            end

            if not self:canConsume() or not self:check("CloseBracket") then
                return false, "esperado ']' aṕos expressão"
            end
            self:consume()

            base = Node.new("IndexAccess", {
                index = expression,
                base = base
            })
        elseif self:check("OpenParen") then
            self:consume()

            local arguments, closed = {}, false
            while self:canConsume() do
                if self:check("Comma") then
                    self:consume()
                elseif self:check("CloseParen") then
                    closed = true
                    break
                end

                local ok, expression = self:parseOr()
                if not ok then
                    return false, expression
                end

                table.insert(arguments, expression)
            end

            if not closed then
                return false, "chamada de funções expera ')'"
            end

            self:consume()

            base = Node.new("CallAccess", {
                arguments = arguments,
                base = base
            })
        else
            break
        end
    end
    return true, base
end
---aqui vai processar os literais como booleanos/números/textos(strings)/referencias
---@return boolean, string|Node
function Parser:parsePrimary()
    if not self:canConsume() then
        return false, "não foi encontrado nenhuma primario"
    end

    --até agora os tokens tinham os dados como texto/string, dado bruto
    --aqui terá conversão de dados
    --"true" para true(booleano), "123" para 123(número)

    local token = self:peek()

    if token.kind == "Minus" then
        self:consume()

        --isso é para inverter o número, transformar de positivo para negativo ou de negativo para positivo
        --exemplo: quando fizer -1 ou fizer -(-1)

        local ok, expression = self:parseOr()
        if not ok then
            return false, expression
        end
        return true, Node.new("UnaryExpression", {expression = expression, operator = '-'})
    elseif token.kind == "Not" then
        self:consume()

        --aqui é parecido como a negação
        --isso é para inverter o booleano, transformar de true para false ou de false para true
        --exemplo: quando fizer not true ou not false

        local ok, expression = self:parseOr()
        if not ok then
            return false, expression
        end
        return true, Node.new("UnaryExpression", {expression = expression, operator = '!'})
    elseif token.kind == "Nil" then
        self:consume()

        --isso é um nulo literal
        return true, Node.new("LiteralNil", token.value == "true")
    elseif token.kind == "Boolean" then
        self:consume()

        --isso é um booleano literal com seu valor convertido para um booleano(de verdade)
        return true, Node.new("LiteralBoolean", token.value == "true")
    elseif token.kind == "Number" then
        self:consume()

        --isso é um número literal com seu valor convertido para um número(de verdade)
        return true, Node.new("LiteralNumber", tonumber(token.value))
    elseif token.kind == "String" then
        self:consume()

        --aqui não necessita converter nada
        return true, Node.new("LiteralString", token.value)
    elseif token.kind == "OpenKey" then
        self:consume()

        local content, closed = {}, false
        while self:canConsume() do
            if self:check("CloseKey") then
                closed = true
                break
            end

            local ok, expression = self:parseOr()
            if ok then
                table.insert(content, expression)

                local symbol = self:peek()
                if symbol.kind == "CloseKey" then
                    closed = true
                    break
                elseif symbol.kind == "Comma" then
                    self:consume()
                else
                    return false, "encontrado símbolo não suportado em tabela literal"
                end
            else
                return false, expression
            end
        end

        if not closed then
            return false, "esperado '}' para fechar a tabela"
        end

        self:consume()
        --aqui não necessita converter nada
        return true, Node.new("LiteralTable", content)
    elseif token.kind == "Function" then
        self:consume()

        if not self:canConsume() or not self:check("OpenParen") then
            return false, "esperado '('"
        end
        self:consume()

        local parameters, closed = {}, false
        while self:canConsume() do
            if self:check("Identifier") then
                table.insert(parameters, self:peek().value)
                self:consume()
            elseif self:check("CloseParen") then
                closed = true
                break
            else
                return false, "esperado um identificador"
            end
        end

        if not closed then
            return false, "declaração de funções expera ')'"
        end
        self:consume()

        if not self:canConsume() or not self:check("OpenKey") then
            return false, "função espera um bloco"
        end

        local ok, block = self:parseBlockStatement()
        if not ok then
            return false, block
        end

        return true, Node.new("LiteralCallable", {parameters = parameters, block = block})
    elseif token.kind == "OpenParen" then
        self:consume()

        if not self:canConsume() then
            return false, "expressão agrupada malformado, experado uma expressão após '('"
        end

        local ok, expression = self:parseOr()
        if not ok then
            return false, expression
        end

        if not self:canConsume() or not self:check("CloseParen") then
            return false, "expressão agrupada malformado, experado ')'"
        end
        self:consume()

        return true, Node.new("GroupedExpression", expression)
    elseif token.kind == "Identifier" then
        self:consume()

        local reference = Node.new("Reference", token.value)
        if not self:canConsume() then
            return true, reference
        end

        if self:check("OpenParen") then
            self:consume()

            local arguments, closed = {}, false
            while self:canConsume() do
                if self:check("Comma") then
                    self:consume()
                elseif self:check("CloseParen") then
                    closed = true
                    break
                end

                local ok, expression = self:parseOr()
                if not ok then
                    return false, expression
                end

                table.insert(arguments, expression)
            end

            if not closed then
                return false, "chamada de funções expera ')'"
            end

            self:consume()

            reference = Node.new("Callable", {
                name = token.value,
                arguments = arguments
            })
        end

        return self:parseAccess(reference)
    end

    return false, "primario não suportado"
end
---aqui vai processar declaração
---@return boolean, string|Node
function Parser:parseDeclareStatement()
    self:consume()

    if not self:canConsume() or not self:check("Identifier") then
        return false, "declaração expera um nome"
    end

    local name = self:peek()
    self:consume()

    local ok, value = self:parseOr()
    if not ok then
        return false, value
    end
    return true, Node.new("DeclareStatement", {name = name.value, value = value})
end
---aqui vai processar declaração de if
---@return boolean, string|Node
function Parser:parseIfStatement()
    self:consume()

    local ok, expression = self:parseOr()
    if not ok then
        return false, expression
    end

    if not self:canConsume() or not self:check("OpenKey") then
        return false, "declaração de if expera um bloco {}"
    end

    local ok, block = self:parseBlockStatement()
    if not ok then
        return false, block
    end

    local ok, unless
    if self:canConsume() and self:check("Else") then
        self:consume()

        if self:canConsume() then
            if self:check("If") then
                ok, unless = self:parseIfStatement()
            elseif self:check("OpenKey") then
                ok, unless = self:parseBlockStatement()
            else
                return false, "após else é esperado if ou {}"
            end
        end

        if not ok then
            return false, unless
        end
    end

    return true, Node.new("IfStatement", {expression = expression, block = block, unless = unless})
end
---aqui vai processar declaração de um bloco de código
---@return boolean, string|Node
function Parser:parseBlockStatement()
    self:consume()

    local content, closed = {}, false
    while self:canConsume() do
        if self:check("CloseKey") then
            closed = true
            break
        end

        local ok, statement = self:parseStatement()
        if not ok then
            return false, statement
        end

        table.insert(content, statement)
    end

    if not closed then
        return false, "bloco expera '}'"
    end

    self:consume()
    return true, Node.new("BlockStatement", content)
end
---declaração de um novo defer
---@return boolean, string|Node
function Parser:parseDeferStatement()
    self:consume()

    local ok, statement = self:parseStatement()
    if not ok then
        return false, statement
    end

    return true, Node.new("DeferStatement", statement)
end
---declaração de um loop for
---@return boolean, string|Node
function Parser:parseForStatement()
    self:consume()
    if not self:canConsume() then
        return false, "loop for expera uma variavel"
    end

    local ok, variable = self:parseDeclareStatement()
    if not ok then
        return false, variable
    end

    if not self:canConsume() or not self:check("Comma") then
        return false, "loop for expera uma ',' para separar variavel e valor máximo"
    end
    self:consume()

    local ok, maximum = self:parseOr()
    if not ok then
        return false, maximum
    end

    if not self:canConsume() or not self:check("Comma") then
        return false, "experado a expressão após valor máximo"
    end
    self:consume()

    local ok, step = self:parsePrimary()
    if not ok then
        return false, step
    end

    if not self:canConsume() or not self:check("OpenKey") then
        return false, "experado um bloco após a expressão do for"
    end

    local ok, block = self:parseBlockStatement()
    if not ok then
        return false, block
    end
    return true, Node.new("ForStatement", {variable = variable, maximum = maximum, step = step, block = block})
end
---declaração de um loop while
---@return boolean, string|Node
function Parser:parseWhileStatement()
    self:consume()

    if not self:canConsume() then
        return false, "experado a expressão após while"
    end

    local ok, expression = self:parseOr()
    if not ok then
        return false, expression
    end

    if not self:canConsume() or not self:check("OpenKey") then
        return false, "experado um bloco após a expressão do while"
    end

    local ok, block = self:parseBlockStatement()
    if not ok then
        return false, block
    end
    return true, Node.new("WhileStatement", {expression = expression, block = block})
end
---aqui é processado declarações como variavel/funções/if/...
---@return boolean, string|Node
function Parser:parseStatement()
    if not self:canConsume() then
        return false, "não foi encontrado nenhuma declaração"
    end

    local token = self:peek()

    if token.kind == "Declare" then
        return self:parseDeclareStatement()
    elseif token.kind == "If" then
        return self:parseIfStatement()
    elseif token.kind == "OpenKey" then
        return self:parseBlockStatement()
    elseif token.kind == "Defer" then
        return self:parseDeferStatement()
    elseif token.kind == "For" then
        return self:parseForStatement()
    elseif token.kind == "While" then
        return self:parseWhileStatement()
    end

    local ok, target = self:parsePrimary()
    if not ok then
        return false, target
    end

    if self:canConsume() and self:check("Assign") then
        self:consume()

        local ok, expression = self:parseOr()
        if not ok then
            return false, expression
        end
        return true, Node.new("AssignmentStatement", {target = target, expression = expression})
    end

    return true, Node.new("StatementExpression", target)
end
---aqui é a porta de entrada processamento dos tokens e terá a criação dos nodes
---@param tokens Token[]
function Parser:parse(tokens)
    self.tokens = tokens

    local content = {}
    while self:canConsume() do
        local ok, result = self:parseStatement()
        if ok then
           table.insert(content, result)
        else
            error(result)
        end
    end

    return Node.new("BlockStatement", content) --esse é o node(nó) raiz, ele é a raiz da árvore
end
return Parser